/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#ifndef _mx_simplempi_h
#define _mx_simplempi_h

#include <stdio.h>
#include <assert.h>
#include "myriexpress.h"

/* private structures and defintions */

struct mxsmpi_comm {
  mx_endpoint_t ep;
  struct mxsmpi_peer {
    mx_endpoint_addr_t addr;
    uint64_t nic;
    char name[256 - sizeof(mx_endpoint_addr_t) - sizeof(uint64_t)];
  } *peers;
  int id;
  int rank;
  int size;
};

extern struct mxsmpi_comm mxsmpi_comm_world;
extern struct mxsmpi_comm mxsmpi_comm_self;

#define MXSMPI_MATCH(commid,rank,tag) (((uint64_t)(uint16_t)(commid) << 48) | ((uint64_t)(uint16_t)(rank) << 32) | (uint32_t)(tag))
#define MXSMPI_MATCH_TAG(match) ((uint32_t)(match))
#define MXSMPI_MATCH_SOURCE(match) ((uint16_t)((match)>>32))

#define MXSMPI_TYPE_ID (0x2428ULL)
#define MXSMPI_TYPE(a,b)  (sizeof(a) | ((uint64_t)MXSMPI_TYPE_ID<<48 | (uint64_t)(b)<<32))
#define MXSMPI_VALID_TYPE(type) ((type >> 48) == MXSMPI_TYPE_ID)

/* MPI definitions -- type and constants */

typedef struct mxsmpi_comm *MXSMPI_Comm;
typedef void MXSMPI_Handler_function(MXSMPI_Comm *, int *, ...); 

typedef MXSMPI_Handler_function *MXSMPI_Errhandler;

typedef struct {
  uint32_t MXSMPI_COUNT;
  uint32_t MXSMPI_TAG;
  uint16_t MXSMPI_SOURCE;
  uint16_t pad;
} MXSMPI_Status;

typedef mx_request_t MXSMPI_Request;
typedef struct { int dummy; } MXSMPI_Group;

typedef uint64_t MXSMPI_Datatype;
#define MXSMPI_COMM_WORLD (&mxsmpi_comm_world)
#define MXSMPI_COMM_SELF (&mxsmpi_comm_self)
#define MXSMPI_SUCCESS 0
#define MXSMPI_UNDEFINED 1
#define MXSMPI_ERR_INTERN 2

#define MXSMPI_PROC_NULL (-1)
#define MXSMPI_REQUEST_NULL (-1)
#define MXSMPI_ANY_SOURCE (-1)
#define MXSMPI_ANY_TAG (-1)

#define MXSMPI_CHAR MXSMPI_TYPE(char,0)
#define MXSMPI_SHORT MXSMPI_TYPE(short,1)
#define MXSMPI_INT MXSMPI_TYPE(int,2)
#define MXSMPI_LONG MXSMPI_TYPE(long,3)

#define MXSMPI_UNSIGNED_CHAR MXSMPI_TYPE(unsigned char,4)
#define MXSMPI_UNSIGNED_SHORT MXSMPI_TYPE(unsigned short,5)
#define MXSMPI_UNSIGNED MXSMPI_TYPE(unsigned,6)
#define MXSMPI_UNSIGNED_LONG MXSMPI_TYPE(unsigned long,7)
#define MXSMPI_FLOAT MXSMPI_TYPE(float,8)
#define MXSMPI_DOUBLE MXSMPI_TYPE(double,9)
#define MXSMPI_DATATYPE_NULL 0

#define MXSMPI_COMM_NULL ((struct mxsmpi_comm *)0)
#define MXSMPI_ERRHANDLER_NULL ((MXSMPI_Errhandler)0)
#define MXSMPI_ERRORS_RETURN ((MXSMPI_Errhandler)0)

#define MXSMPI_NB_TYPES 10

enum mxsmpi_coll_op {
  MXSMPI_OP_NULL,
  MXSMPI_SUM,
  MXSMPI_MAX,
  MXSMPI_MIN,
  MXSMPI_NB_OPS
};

typedef enum mxsmpi_coll_op MXSMPI_Op;

#define MXSMPI_BYTE MXSMPI_CHAR


#define MXSMPI_MAX_PROCESSOR_NAME 256
#define MXSMPI_MAX_ERROR_STRING 256

/* MPI definitions -- prototypes */

int MXSMPI_Ssend(void* buf, int count, MXSMPI_Datatype datatype, int dest, int tag, MXSMPI_Comm comm);
int MXSMPI_Iprobe(int source, int tag, MXSMPI_Comm comm, int *flag, MXSMPI_Status *status);
int MXSMPI_Probe(int source, int tag, MXSMPI_Comm comm, MXSMPI_Status *status);
int MXSMPI_Sendrecv(void *sendbuf, int sendcount, MXSMPI_Datatype sendtype, int dest, int sendtag, void *recvbuf, int recvcount, MXSMPI_Datatype recvtype, int source, MXSMPI_Datatype recvtag, MXSMPI_Comm comm, MXSMPI_Status *status);
int MXSMPI_Type_contiguous(int count, MXSMPI_Datatype oldtype, MXSMPI_Datatype *newtype);
int MXSMPI_Type_commit(MXSMPI_Datatype *type);
int MXSMPI_Type_free(MXSMPI_Datatype *type);

int MXSMPI_Testsome(int incount, MXSMPI_Request in_req[], int *outcount, int idx[], MXSMPI_Status status[]);
int MXSMPI_Waitsome(int incount, MXSMPI_Request in_req[], int *outcount, int idx[], MXSMPI_Status status[]);
int MXSMPI_Waitall(int incount, MXSMPI_Request in_req[], MXSMPI_Status status[]);

int MXSMPI_Init(int *argc, char ***argv);
int MXSMPI_Finalize(void);

int MXSMPI_Barrier(MXSMPI_Comm comm);
int MXSMPI_Bcast(void* buffer, int count, MXSMPI_Datatype datatype, int root, MXSMPI_Comm comm );
int MXSMPI_Reduce(void* sendbuf, void* recvbuf, int count, MXSMPI_Datatype datatype, MXSMPI_Op op, int root, MXSMPI_Comm comm);
int MXSMPI_Allreduce(void* sendbuf, void* recvbuf, int count, MXSMPI_Datatype datatype, MXSMPI_Op op, MXSMPI_Comm comm);
int MXSMPI_Gather(void* sendbuf, int sendcount, MXSMPI_Datatype sendtype, 
		  void* recvbuf, int recvcount, MXSMPI_Datatype recvtype, 
		  int root, MXSMPI_Comm comm);
int MXSMPI_Gatherv(void* sendbuf, int sendcount, MXSMPI_Datatype sendtype, 
		   void* recvbuf, int *recvcounts, int *displs, MXSMPI_Datatype recvtype,
		   int root, MXSMPI_Comm comm);
int MXSMPI_Scatter(void* sendbuf, int sendcount, MXSMPI_Datatype sendtype, 
		   void* recvbuf, int recvcount, MXSMPI_Datatype recvtype, 
		   int root, MXSMPI_Comm comm);
int MXSMPI_Scatterv(void* sendbuf, int *sendcounts, int *displs, MXSMPI_Datatype sendtype, 
		    void* recvbuf, int recvcount, MXSMPI_Datatype recvtype,
		    int root, MXSMPI_Comm comm);


int MXSMPI_Type_size(MXSMPI_Datatype datatype, int *size);

double MXSMPI_Wtime(void);
double MXSMPI_Wtick(void);
int MXSMPI_Abort(MXSMPI_Comm comm, int);
int MXSMPI_Get_processor_name(char *name, int *resultlen);
int MXSMPI_Errhandler_create(MXSMPI_Handler_function *function, MXSMPI_Errhandler *errhandler);
int MXSMPI_Errhandler_set(MXSMPI_Comm comm, MXSMPI_Errhandler errhandler);
int MXSMPI_Errhandler_free(MXSMPI_Errhandler *errhandler);
int MXSMPI_Error_string(int errorcode, char *string, int *resultlen);

int MXSMPI_Comm_dup(MXSMPI_Comm comm, MXSMPI_Comm *dup);

/* MPI definitions -- inline */
static inline int MXSMPI_Isend(void* buf, int count, MXSMPI_Datatype datatype, int dest, int tag, MXSMPI_Comm comm, MXSMPI_Request *request)
{
  mx_segment_t seg;
  assert((datatype >> 48) == MXSMPI_TYPE_ID);
  seg.segment_ptr = buf;
  seg.segment_length = count*(uint32_t)datatype;
  mx_isend(comm->ep,&seg,1,comm->peers[dest].addr, MXSMPI_MATCH(comm->id,comm->rank,tag),NULL, request);
  return MXSMPI_SUCCESS;
}

static inline int MXSMPI_Issend(void* buf, int count, MXSMPI_Datatype datatype, int dest, int tag, MXSMPI_Comm comm, MXSMPI_Request *request)
{
  mx_segment_t seg;
  assert((datatype >> 48) == MXSMPI_TYPE_ID);
  seg.segment_ptr = buf;
  seg.segment_length = count*(uint32_t)datatype;
  mx_issend(comm->ep,&seg,1,comm->peers[dest].addr, MXSMPI_MATCH(comm->id,comm->rank,tag),NULL, request);
  return MXSMPI_SUCCESS;
}

static inline int MXSMPI_Send(void* buf, int count, MXSMPI_Datatype datatype, int dest, int tag, MXSMPI_Comm comm)
{
  mx_request_t req;
  mx_segment_t seg;
  uint32_t res;
  mx_status_t status;
  assert(MXSMPI_VALID_TYPE(datatype));
  seg.segment_ptr = buf;
  seg.segment_length = count*(uint32_t)datatype;
  mx_isend(comm->ep,&seg,1,comm->peers[dest].addr, MXSMPI_MATCH(comm->id,comm->rank,tag),NULL, &req);
  mx_wait(comm->ep,&req,MX_INFINITE,&status,&res);
  return MXSMPI_SUCCESS;
}


static inline int MXSMPI_Recv(void* buf, int count, MXSMPI_Datatype datatype, int source, int tag, MXSMPI_Comm comm, MXSMPI_Status *status)
{
  mx_request_t req;
  mx_segment_t seg;
  uint32_t res;
  uint64_t mask;
  mx_status_t s;
  assert(MXSMPI_VALID_TYPE(datatype));
  seg.segment_ptr = buf;
  seg.segment_length = count*(uint32_t)datatype;
  mask = MXSMPI_MATCH(-1,~source >> 16,~tag >> 31);
  mx_irecv(comm->ep,&seg,1,MXSMPI_MATCH(comm->id,source,tag) & mask, mask,NULL, &req);
  mx_wait(comm->ep,&req,MX_INFINITE,&s,&res);
  status->MXSMPI_SOURCE = MXSMPI_MATCH_SOURCE(s.match_info);
  status->MXSMPI_TAG = MXSMPI_MATCH_TAG(s.match_info);
  status->MXSMPI_COUNT = s.msg_length;
  return MXSMPI_SUCCESS;
}

static inline int MXSMPI_Irecv(void* buf, int count, MXSMPI_Datatype datatype, int source, int tag, MXSMPI_Comm comm, MXSMPI_Request *request)
{
  mx_segment_t seg;
  uint64_t mask;
  assert((datatype >> 48) == MXSMPI_TYPE_ID);
  seg.segment_ptr = buf;
  seg.segment_length = count*(uint32_t)datatype;
  mask = MXSMPI_MATCH(-1,~source >> 16,~tag >> 31);
  mx_irecv(comm->ep,&seg,1,MXSMPI_MATCH(comm->id,source,tag)&mask,mask,NULL, request);
  return MXSMPI_SUCCESS;
}

static inline int MXSMPI_Wait(MXSMPI_Request *request, MXSMPI_Status *status)
{
  mx_status_t s;
  uint32_t res;
  mx_wait(MXSMPI_COMM_WORLD->ep,request,MX_INFINITE,&s,&res);
  status->MXSMPI_SOURCE = MXSMPI_MATCH_SOURCE(s.match_info);
  status->MXSMPI_TAG = MXSMPI_MATCH_TAG(s.match_info);
  status->MXSMPI_COUNT = s.msg_length;
  return MXSMPI_SUCCESS;
}


static inline int MXSMPI_Test(MXSMPI_Request *request, int *flag, MXSMPI_Status *status)
{
  mx_status_t s;
  mx_test(MXSMPI_COMM_WORLD->ep,request,&s,(uint32_t*)flag);
  status->MXSMPI_SOURCE = MXSMPI_MATCH_SOURCE(s.match_info);
  status->MXSMPI_TAG = MXSMPI_MATCH_TAG(s.match_info);
  return MXSMPI_SUCCESS;
}

static int inline MXSMPI_Comm_size(MXSMPI_Comm comm, int *size)
{
  *size = comm->size;
  return MXSMPI_SUCCESS;
}

static int inline MXSMPI_Comm_rank(MXSMPI_Comm comm, int *rank)
{
  *rank = comm->rank;
  return MXSMPI_SUCCESS;
}

static int inline MXSMPI_Get_count(MXSMPI_Status *status, MXSMPI_Datatype datatype, int *count)
{
  *count = status->MXSMPI_COUNT / (uint32_t)datatype;
  return MXSMPI_SUCCESS;
}

#endif
